<?php
session_start();
include_once("config.php");
include_once("functions.php");

cek_login_user();

if (!isset($_SESSION['user_id'])) {
    header("Location: masuk.php");
    exit();
}

// Cek apakah ada data POST dan session transaksi
if ($_SERVER["REQUEST_METHOD"] != "POST" || !isset($_SESSION['transaksi'])) {
    header("Location: index.php");
    exit();
}

$kode_transaksi = isset($_POST['kode_transaksi']) ? htmlspecialchars($_POST['kode_transaksi']) : '';
$game_type = isset($_POST['game_type']) ? htmlspecialchars($_POST['game_type']) : 'ml';

// Ambil data transaksi dari session
$transaksi = $_SESSION['transaksi'];

// Verifikasi kode transaksi
if ($kode_transaksi !== $transaksi['kode']) {
    header("Location: index.php");
    exit();
}

// Mapping nama currency berdasarkan game type
$currency_names = [
    'ml' => 'Diamond',
    'pubg' => 'UC',
    'genshin' => 'Genesis Crystal',
    'honor' => 'Token',
    'valorant' => 'Valorant Points',
    'ff' => 'Diamond'
];

$currency_name = isset($currency_names[$game_type]) ? $currency_names[$game_type] : 'Item';

// Fungsi untuk menyimpan data transaksi ke database - FIXED VERSION
function simpanDataTransaksi($transaksi) {
    global $conn;
    
    try {
        // Fixed SQL - 12 columns, 12 placeholders
        $sql = "INSERT INTO transaksi (
            kode_transaksi, 
            user_id, 
            userid_game, 
            zoneid_game, 
            nama_game,
            item_jenis,
            nominal,
            total_bayar, 
            metode_pembayaran, 
            whatsapp, 
            tanggal, 
            status
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $conn->prepare($sql);
        
        // Check if prepare() was successful
        if ($stmt === false) {
            error_log("SQL Prepare Error: " . $conn->error);
            error_log("SQL Query: " . $sql);
            return false;
        }
        
        // Ekstrak item dari nominal
        $nominal_text = preg_replace('/Rp\s+[\d.,]+/', '', $transaksi['nominal']);
        $item_parts = explode(' ', trim($nominal_text));
        $item_jumlah = isset($item_parts[0]) ? $item_parts[0] : '';
        $item_jenis = isset($item_parts[1]) ? $item_parts[1] : '';
        
        $total_bayar_clean = floatval($transaksi['total_bayar']);
        
        // Fixed bind_param - 12 parameters matching 12 placeholders
        $result = $stmt->bind_param(
            "sssssssdssss", // 12 type specifiers (s=string, d=double/decimal)
            $transaksi['kode'],
            $transaksi['user_id'],
            $transaksi['userid'],
            $transaksi['zoneid'],
            $transaksi['game_name'],
            $item_jenis,
            $transaksi['nominal'],
            $total_bayar_clean,
            $transaksi['metode_nama'],
            $transaksi['whatsapp'],
            $transaksi['tanggal'],
            $transaksi['status']
        );
        
        if (!$result) {
            error_log("SQL Bind Error: " . $stmt->error);
            $stmt->close();
            return false;
        }
        
        $execute_result = $stmt->execute();
        
        if (!$execute_result) {
            error_log("SQL Execute Error: " . $stmt->error);
            $stmt->close();
            return false;
        }
        
        $stmt->close();
        return true;
        
    } catch (Exception $e) {
        error_log("Error menyimpan transaksi: " . $e->getMessage());
        return false;
    }
}

// Fungsi untuk mengecek status transaksi dari database
function cekStatusTransaksi($kode_transaksi) {
    global $conn;
    
    try {
        $sql = "SELECT status FROM transaksi WHERE kode_transaksi = ? ORDER BY tanggal DESC LIMIT 1";
        $stmt = $conn->prepare($sql);
        
        if ($stmt === false) {
            error_log("SQL Prepare Error: " . $conn->error);
            return false;
        }
        
        $stmt->bind_param("s", $kode_transaksi);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $row = $result->fetch_assoc();
            $stmt->close();
            return $row['status'];
        }
        
        $stmt->close();
        return false;
        
    } catch (Exception $e) {
        error_log("Error mengecek status transaksi: " . $e->getMessage());
        return false;
    }
}

// PERBAIKAN: Tentukan status berdasarkan kondisi pembayaran
// Cek apakah ini request refresh status
if (isset($_POST['refresh_status']) && $_POST['refresh_status'] === 'true') {
    // Cek status terbaru dari database
    $status_db = cekStatusTransaksi($transaksi['kode']);
    if ($status_db !== false) {
        $status_transaksi = $status_db;
    } else {
        $status_transaksi = 'pending'; // Default jika belum ada di database
    }
} else {
    // Logika status untuk transaksi baru
    if (isset($_POST['payment_verified']) && $_POST['payment_verified'] === 'true') {
        $status_transaksi = 'success';
    } else {
        $status_transaksi = 'pending';
    }
}

// Siapkan data transaksi untuk disimpan (hanya jika belum ada di database)
if (!isset($_POST['refresh_status'])) {
    $data_transaksi = [
        'kode' => $transaksi['kode'],
        'user_id' => $_SESSION['user_id'],
        'userid' => $transaksi['userid'],
        'zoneid' => $transaksi['zoneid'],
        'nominal' => $transaksi['nominal'], // Data item/nominal
        'total_bayar' => floatval($transaksi['harga']), // PERBAIKAN: Pastikan format numeric
        'metode' => $transaksi['metode'],
        'metode_nama' => $transaksi['metode_nama'],
        'whatsapp' => $transaksi['whatsapp'],
        'tanggal' => $transaksi['tanggal'],
        'game_type' => $transaksi['game_type'],
        'game_name' => $transaksi['game_name'],
        'status' => $status_transaksi // PERBAIKAN: Gunakan status yang sudah ditentukan
    ];

    // Simpan data transaksi ke database
    $simpan_berhasil = simpanDataTransaksi($data_transaksi);

    if (!$simpan_berhasil) {
        error_log("Gagal menyimpan transaksi: " . $transaksi['kode']);
        // Optional: Anda bisa menambahkan handling error di sini
        // Misalnya redirect ke halaman error atau tampilkan pesan
    }
}

// Bersihkan session transaksi setelah berhasil (kecuali jika refresh)
if (!isset($_POST['refresh_status'])) {
    // Simpan data transaksi ke session untuk keperluan refresh
    $_SESSION['transaksi_temp'] = $transaksi;
} else {
    // Gunakan data dari session temp untuk refresh
    if (isset($_SESSION['transaksi_temp'])) {
        $transaksi = $_SESSION['transaksi_temp'];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title><?php echo ($status_transaksi === 'success') ? 'Pembayaran Berhasil' : 'Pembayaran Pending'; ?> - KMGSTORE</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <style>
    body {
      margin: 0;
      font-family: Arial, sans-serif;
      background: #1e1e1e;
      color: #fff;
    }
    .navbar {
      position: relative;
      background: #1e2635;
      display: flex;
      align-items: center;
      justify-content: space-between;
      padding: 10px 20px;
      z-index: 5;
      backdrop-filter: blur(10px);
    }
    .container {
      max-width: 600px;
      margin: auto;
      padding: 20px;
    }
    .status-box {
      background: rgba(42, 42, 42, 0.8);
      backdrop-filter: blur(10px);
      padding: 30px;
      border-radius: 15px;
      margin-top: 20px;
      text-align: center;
    }
    .status-icon {
      width: 80px;
      height: 80px;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      margin: 0 auto 20px auto;
      font-size: 40px;
    }
    .success-icon {
      background: #10b981;
    }
    .pending-icon {
      background: #f59e0b;
    }
    .failed-icon {
      background: #ef4444;
    }
    .transaction-details {
      background: rgba(0, 0, 0, 0.2);
      padding: 20px;
      border-radius: 10px;
      margin: 20px 0;
      text-align: left;
    }
    .detail-row {
      display: flex;
      justify-content: space-between;
      padding: 10px 0;
      border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    }
    .detail-row:last-child {
      border-bottom: none;
    }
    .btn {
      background: goldenrod;
      border: none;
      padding: 15px 30px;
      color: rgb(85, 32, 32);
      font-weight: bold;
      font-size: 16px;
      border-radius: 8px;
      cursor: pointer;
      margin: 10px;
      text-decoration: none;
      display: inline-block;
    }
    .btn-secondary {
      background: rgba(255, 255, 255, 0.1);
      color: white;
    }
    .btn-refresh {
      background: #3b82f6;
      color: white;
    }
    .btn-refresh:hover {
      background: #2563eb;
    }
    .btn-refresh:disabled {
      background: #6b7280;
      cursor: not-allowed;
    }
    .footer-bottom {
      background-color: #1e2635;
      color: white;
      padding: 20px;
      text-align: center;
      font-size: 13px;
      margin-top: 40px;
    }
    .animate-bounce {
      animation: bounce 1s infinite;
    }
    .animate-spin {
      animation: spin 1s linear infinite;
    }
    @keyframes bounce {
      0%, 20%, 53%, 80%, 100% {
        transform: translate3d(0,0,0);
      }
      40%, 43% {
        transform: translate3d(0,-30px,0);
      }
      70% {
        transform: translate3d(0,-15px,0);
      }
      90% {
        transform: translate3d(0,-4px,0);
      }
    }
    @keyframes spin {
      from {
        transform: rotate(0deg);
      }
      to {
        transform: rotate(360deg);
      }
    }
    .status-changed {
      animation: statusChange 0.5s ease-in-out;
    }
    @keyframes statusChange {
      0% {
        transform: scale(1);
      }
      50% {
        transform: scale(1.05);
      }
      100% {
        transform: scale(1);
      }
    }
  </style>
</head>
<body>

  <!-- Navbar -->
    <nav class="bg-gray-800 bg-opacity-80 backdrop-filter backdrop-blur-md sticky top-0 z-50 p-4 flex justify-between items-center shadow-lg">
        <!-- Kiri: Logo -->
        <div class="flex items-center space-x-3">
            <div class="relative">
                <!-- Perbaikan path gambar logo -->
                <img src="../img/logo-tokioiboy.jpg" alt="Logo Tokiboy Store" class="h-10 w-10 rounded-full border-2 border-yellow-500" />
                <div class="absolute -top-1 -right-1 h-3 w-3 bg-green-500 rounded-full"></div>
            </div>
            <div>
                <span class="text-xl font-bold bg-gradient-to-r from-yellow-400 via-yellow-300 to-yellow-500 text-transparent bg-clip-text">KMGSTORE</span>
                <p class="text-xs text-gray-400">Top Up Termurah & Tercepat</p>
            </div>
        </div>

        <!-- Kanan: Menu / User -->
        <div class="space-x-2 flex items-center">
            <?php if (isset($_SESSION['nama'])): ?>
                <div class="bg-gray-700 px-3 py-1 rounded-full flex items-center">
                    <i class="fas fa-user-circle mr-2"></i>
                    <span class="text-white">Halo, <?= htmlspecialchars($_SESSION['nama']) ?>!</span>
                </div>
                <a href="../index.php" class="bg-red-500 hover:bg-red-600 text-white font-bold py-2 px-4 rounded-full flex items-center">
                    <i class="fas fa-home mr-1"></i> Home
                </a>
            <?php else: ?>
                <a href="masuk.php" class="bg-yellow-500 text-black px-3 py-1 rounded">Masuk</a>
                <a href="daftar.php" class="bg-gray-700 px-3 py-1 rounded border border-yellow-500">Daftar</a>
            <?php endif; ?>
        </div>
    </nav>

  <div class="container">
    <div class="status-box">
      <?php if ($status_transaksi === 'success'): ?>
        <div class="status-icon success-icon animate-bounce">
          ✓
        </div>
        <h1 class="text-3xl font-bold text-green-400 mb-4">Pembayaran Berhasil!</h1>
        <p class="text-lg mb-6">Terima kasih! Transaksi Anda telah berhasil diproses.</p>
      <?php elseif ($status_transaksi === 'pending'): ?>
        <div class="status-icon pending-icon animate-bounce">
          ⏳
        </div>
        <h1 class="text-3xl font-bold text-yellow-400 mb-4">Pembayaran Pending</h1>
        <p class="text-lg mb-6">Transaksi Anda sedang diproses, mohon tunggu konfirmasi pembayaran.</p>
      <?php else: ?>
        <div class="status-icon failed-icon animate-bounce">
          ✗
        </div>
        <h1 class="text-3xl font-bold text-red-400 mb-4">Pembayaran Gagal</h1>
        <p class="text-lg mb-6">Maaf, transaksi Anda gagal diproses. Silakan coba lagi.</p>
      <?php endif; ?>
      
      <div class="transaction-details">
        <h3 class="text-xl font-bold mb-4 text-center">Detail Transaksi</h3>
        
        <div class="detail-row">
          <div class="font-semibold">ID Transaksi</div>
          <div><?= htmlspecialchars($transaksi['kode']) ?></div>
        </div>
        
        <div class="detail-row">
          <div class="font-semibold">Tanggal</div>
          <div><?= date('d-m-Y H:i', strtotime($transaksi['tanggal'])) ?></div>
        </div>
        
        <div class="detail-row">
          <div class="font-semibold">Game</div>
          <div><?= htmlspecialchars($transaksi['game_name']) ?></div>
        </div>
        
        <div class="detail-row">
          <div class="font-semibold">User ID</div>
          <div><?= htmlspecialchars($transaksi['userid']) ?><?= !empty($transaksi['zoneid']) ? " (" . htmlspecialchars($transaksi['zoneid']) . ")" : "" ?></div>
        </div>
        
        <div class="detail-row">
          <div class="font-semibold">Item</div>
          <div><?php
            $nominal_text = preg_replace('/Rp\s+[\d.,]+/', '', $transaksi['nominal']);
            echo htmlspecialchars(trim($nominal_text));
          ?></div>
        </div>
        
        <div class="detail-row">
          <div class="font-semibold">Metode Pembayaran</div>
          <div><?= htmlspecialchars($transaksi['metode_nama']) ?></div>
        </div>
        
        <div class="detail-row">
          <div class="font-semibold">Total Dibayar</div>
          <div class="text-green-400 font-bold">Rp <?= number_format(intval($transaksi['harga']), 0, ',', '.') ?></div>
        </div>
        
        <div class="detail-row">
          <div class="font-semibold">Status</div>
          <div class="<?php 
            echo ($status_transaksi === 'success') ? 'text-green-400' : 
                 (($status_transaksi === 'pending') ? 'text-yellow-400' : 'text-red-400'); 
          ?> font-bold" id="status-display">
            <?php 
              echo ($status_transaksi === 'success') ? '✓ BERHASIL' : 
                   (($status_transaksi === 'pending') ? '⏳ PENDING' : '✗ GAGAL'); 
            ?>
          </div>
        </div>
      </div>
      
      <?php if ($status_transaksi === 'success'): ?>
        <div class="bg-green-900 bg-opacity-30 border border-green-500 rounded-lg p-4 mt-6">
          <p class="text-green-300 font-semibold">🎉 <?= htmlspecialchars($currency_name) ?> berhasil ditambahkan ke akun game Anda!</p>
          <p class="text-sm text-green-200 mt-2">Silakan cek akun game Anda dalam 1-5 menit.</p>
        </div>
      <?php elseif ($status_transaksi === 'pending'): ?>
        <div class="bg-yellow-900 bg-opacity-30 border border-yellow-500 rounded-lg p-4 mt-6">
          <p class="text-yellow-300 font-semibold">⏳ Transaksi sedang diproses</p>
          <p class="text-sm text-yellow-200 mt-2">Mohon tunggu, <?= htmlspecialchars($currency_name) ?> akan ditambahkan ke akun game Anda setelah pembayaran dikonfirmasi.</p>
          
          <!-- Tombol Refresh Status -->
          <form method="POST" action="" id="refresh-form" class="mt-4">
            <input type="hidden" name="kode_transaksi" value="<?= htmlspecialchars($transaksi['kode']) ?>">
            <input type="hidden" name="game_type" value="<?= htmlspecialchars($game_type) ?>">
            <input type="hidden" name="refresh_status" value="true">
            <button type="submit" class="btn btn-refresh" id="refresh-btn">
              <span id="refresh-icon">🔄</span>
              <span id="refresh-text">Cek Status Pembayaran</span>
            </button>
          </form>
          
          <p class="text-xs text-yellow-100 mt-2">Klik tombol di atas untuk mengecek apakah admin sudah mengonfirmasi pembayaran Anda.</p>
        </div>
      <?php endif; ?>
      
      <?php if (!empty($transaksi['whatsapp'])): ?>
      <div class="bg-blue-900 bg-opacity-30 border border-blue-500 rounded-lg p-4 mt-4">
        <p class="text-blue-300">📱 Struk pembayaran telah dikirim ke WhatsApp: <?= htmlspecialchars($transaksi['whatsapp']) ?></p>
      </div>
      <?php endif; ?>
      
      <div class="mt-8">
        <a href="../index.php" class="btn">Kembali ke Beranda</a>
        <button onclick="window.print()" class="btn btn-secondary">Cetak Struk</button>
      </div>
      
      <div class="mt-6 text-sm text-gray-400">
        <p>Butuh bantuan? Hubungi customer service kami:</p>
        <p>WhatsApp: <a href="https://wa.me/6281234567890" class="text-blue-400">+62 812-3456-7890</a></p>
      </div>
    </div>
  </div>

  <div class="footer-bottom">© 2025 KMGSTORE. All Rights Reserved.</div>

  <script>
    // Handle refresh form submission
    document.getElementById('refresh-form').addEventListener('submit', function(e) {
      const refreshBtn = document.getElementById('refresh-btn');
      const refreshIcon = document.getElementById('refresh-icon');
      const refreshText = document.getElementById('refresh-text');
      
      // Disable button dan ubah tampilan
      refreshBtn.disabled = true;
      refreshIcon.innerHTML = '⏳';
      refreshIcon.classList.add('animate-spin');
      refreshText.textContent = 'Mengecek Status...';
      
      // Form akan submit secara normal
    });
    
    // Auto refresh setiap 30 detik jika status masih pending
    <?php if ($status_transaksi === 'pending'): ?>
    let autoRefreshInterval;
    
    function startAutoRefresh() {
      autoRefreshInterval = setInterval(function() {
        const form = document.getElementById('refresh-form');
        if (form) {
          form.submit();
        }
      }, 30000); // 30 detik
    }
    
    function stopAutoRefresh() {
      if (autoRefreshInterval) {
        clearInterval(autoRefreshInterval);
      }
    }
    
    // Mulai auto refresh saat halaman dimuat
    window.addEventListener('load', startAutoRefresh);
    
    // Hentikan auto refresh saat pengguna meninggalkan halaman
    window.addEventListener('beforeunload', stopAutoRefresh);
    
    // Tampilkan countdown untuk refresh berikutnya
    let countdown = 30;
    function updateCountdown() {
      const countdownElement = document.getElementById('countdown');
      if (countdownElement) {
        countdownElement.textContent = countdown;
        countdown--;
        if (countdown < 0) {
          countdown = 30;
        }
      }
    }
    
    // Update countdown setiap detik
    setInterval(updateCountdown, 1000);
    
    // Tambahkan elemen countdown ke halaman
    window.addEventListener('load', function() {
      const refreshForm = document.getElementById('refresh-form');
      if (refreshForm) {
        const countdownDiv = document.createElement('div');
        countdownDiv.innerHTML = '<p class="text-xs text-yellow-100 mt-2">Auto refresh dalam: <span id="countdown">30</span> detik</p>';
        refreshForm.appendChild(countdownDiv);
      }
    });
    <?php endif; ?>
    
    // Konfetti effect hanya untuk status success
    <?php if ($status_transaksi === 'success'): ?>
    function createConfetti() {
      const colors = ['#ff6b6b', '#4ecdc4', '#45b7d1', '#96ceb4', '#ffeaa7'];
      
      for (let i = 0; i < 50; i++) {
        const confetti = document.createElement('div');
        confetti.style.position = 'fixed';
        confetti.style.left = Math.random() * 100 + 'vw';
        confetti.style.top = '-10px';
        confetti.style.width = '10px';
        confetti.style.height = '10px';
        confetti.style.backgroundColor = colors[Math.floor(Math.random() * colors.length)];
        confetti.style.opacity = Math.random();
        confetti.style.zIndex = '9999';
        confetti.style.pointerEvents = 'none';
        
        document.body.appendChild(confetti);
        
        const fall = confetti.animate([
          { transform: 'translateY(-10px) rotateZ(0deg)' },
          { transform: `translateY(100vh) rotateZ(${Math.random() * 360}deg)` }
        ], {
          duration: Math.random() * 2000 + 1000,
          easing: 'linear'
        });
        
        fall.addEventListener('finish', () => {
          confetti.remove();
        });
      }
    }
    
    // Jalankan konfetti saat halaman dimuat
    window.addEventListener('load', createConfetti);
    
    // Hentikan auto refresh jika sudah success
    stopAutoRefresh();
    <?php endif; ?>
    
    // Animasi perubahan status
    <?php if (isset($_POST['refresh_status'])): ?>
    window.addEventListener('load', function() {
      const statusDisplay = document.getElementById('status-display');
      if (statusDisplay) {
        statusDisplay.classList.add('status-changed');
        setTimeout(() => {
          statusDisplay.classList.remove('status-changed');
        }, 500);
      }
    });
    <?php endif; ?>
  </script>

</body>
</html>