<?php
session_start();

include 'config.php'; 

// Function untuk validasi email
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

// Function untuk validasi nomor HP
function validatePhone($phone) {
    // Hapus semua karakter non-digit
    $phone = preg_replace('/[^0-9]/', '', $phone);
    
    // Cek apakah dimulai dengan 0 dan memiliki panjang yang sesuai
    if (preg_match('/^0[0-9]{9,12}$/', $phone)) {
        return $phone;
    }
    return false;
}

// Function untuk validasi password
function validatePassword($password) {
    // Minimal 8 karakter, mengandung huruf besar, kecil, angka, dan simbol
    if (strlen($password) < 8) {
        return "Password minimal 8 karakter";
    }
    if (!preg_match('/[a-z]/', $password)) {
        return "Password harus mengandung huruf kecil";
    }
    if (!preg_match('/[A-Z]/', $password)) {
        return "Password harus mengandung huruf besar";
    }
    if (!preg_match('/[0-9]/', $password)) {
        return "Password harus mengandung angka";
    }
    if (!preg_match('/[^A-Za-z0-9]/', $password)) {
        return "Password harus mengandung simbol";
    }
    return true;
}

// Function untuk sanitasi input
function sanitizeInput($data) {
    return htmlspecialchars(strip_tags(trim($data)));
}

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $errors = [];
    
    // Validasi dan sanitasi input
    $nama = isset($_POST['nama']) ? sanitizeInput($_POST['nama']) : '';
    $email = isset($_POST['email']) ? sanitizeInput($_POST['email']) : '';
    $nohp = isset($_POST['nohp']) ? sanitizeInput($_POST['nohp']) : '';
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    $confirmPassword = isset($_POST['confirmPassword']) ? $_POST['confirmPassword'] : '';
    
    // Validasi field required
    if (empty($nama)) {
        $errors[] = "Nama lengkap wajib diisi";
    } elseif (strlen($nama) < 2) {
        $errors[] = "Nama minimal 2 karakter";
    }
    
    if (empty($email)) {
        $errors[] = "Email wajib diisi";
    } elseif (!validateEmail($email)) {
        $errors[] = "Format email tidak valid";
    }
    
    if (empty($nohp)) {
        $errors[] = "Nomor HP wajib diisi";
    } else {
        $validatedPhone = validatePhone($nohp);
        if (!$validatedPhone) {
            $errors[] = "Format nomor HP tidak valid";
        } else {
            $nohp = $validatedPhone;
        }
    }
    
    if (empty($password)) {
        $errors[] = "Password wajib diisi";
    } else {
        $passwordValidation = validatePassword($password);
        if ($passwordValidation !== true) {
            $errors[] = $passwordValidation;
        }
    }
    
    if (empty($confirmPassword)) {
        $errors[] = "Konfirmasi password wajib diisi";
    } elseif ($password !== $confirmPassword) {
        $errors[] = "Password dan konfirmasi password tidak cocok";
    }
    
    // Jika ada error, simpan ke session dan redirect kembali
    if (!empty($errors)) {
        $_SESSION['errors'] = $errors;
        $_SESSION['form_data'] = $_POST;
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    // Cek koneksi database
    if (!$conn) {
        $_SESSION['errors'] = ["Koneksi database gagal. Silakan coba lagi nanti."];
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    // Gunakan prepared statement untuk keamanan
    $check_email = "SELECT id FROM users WHERE email = ?";
    $stmt = mysqli_prepare($conn, $check_email);
    
    if (!$stmt) {
        $_SESSION['errors'] = ["Terjadi kesalahan sistem. Silakan coba lagi."];
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    mysqli_stmt_bind_param($stmt, "s", $email);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if (mysqli_num_rows($result) > 0) {
        $_SESSION['errors'] = ["Email sudah terdaftar! Silakan gunakan email lain."];
        $_SESSION['form_data'] = $_POST;
        mysqli_stmt_close($stmt);
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    mysqli_stmt_close($stmt);
    
    // Cek nomor HP duplikat
    $check_phone = "SELECT id FROM users WHERE nohp = ?";
    $stmt = mysqli_prepare($conn, $check_phone);
    
    if (!$stmt) {
        $_SESSION['errors'] = ["Terjadi kesalahan sistem. Silakan coba lagi."];
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    mysqli_stmt_bind_param($stmt, "s", $nohp);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if (mysqli_num_rows($result) > 0) {
        $_SESSION['errors'] = ["Nomor HP sudah terdaftar! Silakan gunakan nomor lain."];
        $_SESSION['form_data'] = $_POST;
        mysqli_stmt_close($stmt);
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    mysqli_stmt_close($stmt);
    
    // Hash password dengan cost yang lebih tinggi
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT, ['cost' => 12]);
    
    // Insert user baru dengan prepared statement
    $insert_sql = "INSERT INTO users (nama, email, nohp, password, created_at, updated_at) VALUES (?, ?, ?, ?, NOW(), NOW())";
    $stmt = mysqli_prepare($conn, $insert_sql);
    
    if (!$stmt) {
        $_SESSION['errors'] = ["Terjadi kesalahan sistem. Silakan coba lagi."];
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    mysqli_stmt_bind_param($stmt, "ssss", $nama, $email, $nohp, $hashedPassword);
    
    if (mysqli_stmt_execute($stmt)) {
        // Registrasi berhasil
        mysqli_stmt_close($stmt);
        mysqli_close($conn);
        
        // Set session untuk success message
        $_SESSION['registration_success'] = true;
        $_SESSION['success_message'] = "Pendaftaran berhasil! Selamat datang di KMGSTORE, " . $nama . "!";
        $_SESSION['user_name'] = $nama;
        
        // Clear form data and errors
        unset($_SESSION['form_data']);
        unset($_SESSION['errors']);
        
        // Redirect ke halaman yang sama dengan parameter success
        header("Location: " . $_SERVER['PHP_SELF'] . "?success=1");
        exit;
    } else {
        $_SESSION['errors'] = ["Terjadi kesalahan saat mendaftar. Silakan coba lagi."];
        $_SESSION['form_data'] = $_POST;
        mysqli_stmt_close($stmt);
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
}

// Check for success parameter
$show_success = false;
if (isset($_GET['success']) && $_GET['success'] == '1') {
    $show_success = true;
}

// Ambil data error dan form dari session
$errors = isset($_SESSION['errors']) ? $_SESSION['errors'] : [];
$form_data = isset($_SESSION['form_data']) ? $_SESSION['form_data'] : [];

// Check for success message
$success_message = isset($_SESSION['success_message']) ? $_SESSION['success_message'] : '';
$user_name = isset($_SESSION['user_name']) ? $_SESSION['user_name'] : '';

// Hapus data session setelah digunakan (kecuali untuk success message yang akan dihapus setelah ditampilkan)
unset($_SESSION['errors']);
unset($_SESSION['form_data']);
if ($show_success) {
    unset($_SESSION['success_message']);
    unset($_SESSION['user_name']);
    unset($_SESSION['registration_success']);
}
?>

<!DOCTYPE html>
<html lang="id">
  <head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>Daftar - KMGSTORE | Top Up Games Termurah & Tercepat</title>
    <link
      href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css"
      rel="stylesheet"
    />
    <link
      rel="stylesheet"
      href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css"
    />
    <link
      rel="stylesheet"
      href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css"
    />
    <style>
      @import url("https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap");

      body {
        font-family: "Poppins", sans-serif;
        background-color: #0f172a;
        background-image: radial-gradient(
            circle at 25% 25%,
            rgba(66, 26, 91, 0.2) 0%,
            transparent 40%
          ),
          radial-gradient(
            circle at 75% 75%,
            rgba(25, 84, 123, 0.2) 0%,
            transparent 40%
          );
        background-attachment: fixed;
      }

      .gradient-border {
        position: relative;
        border-radius: 0.75rem;
        z-index: 0;
        overflow: hidden;
      }

      .gradient-border::before {
        content: "";
        position: absolute;
        top: -2px;
        left: -2px;
        right: -2px;
        bottom: -2px;
        background: linear-gradient(45deg, #ff8a00, #dd356e, #8a49ff, #3b82f6);
        z-index: -1;
        animation: borderAnimation 8s linear infinite;
        border-radius: 0.85rem;
      }

      @keyframes borderAnimation {
        0% {
          background-position: 0% 50%;
        }
        50% {
          background-position: 100% 50%;
        }
        100% {
          background-position: 0% 50%;
        }
      }

      .glow-btn {
        position: relative;
        overflow: hidden;
        transition: all 0.3s ease;
      }

      .glow-btn:hover {
        box-shadow: 0 0 15px rgba(255, 204, 0, 0.6);
      }

      .glow-btn::after {
        content: "";
        position: absolute;
        top: -50%;
        left: -50%;
        width: 200%;
        height: 200%;
        background: rgba(255, 255, 255, 0.2);
        transform: rotate(45deg);
        opacity: 0;
        transition: all 0.5s;
      }

      .glow-btn:hover::after {
        opacity: 1;
        transform: rotate(45deg) translateX(100%);
      }

      .register-card {
        transition: all 0.3s ease;
        transform: translateY(0);
      }

      .register-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.3);
      }

      .input-group {
        position: relative;
        margin-bottom: 1.5rem;
      }

      .input-group input {
        width: 100%;
        padding: 0.75rem 1rem;
        padding-left: 2.5rem;
        border-radius: 0.5rem;
        background-color: rgba(31, 41, 55, 0.8);
        color: white;
        border: 1px solid rgba(75, 85, 99, 0.5);
        transition: all 0.3s ease;
      }

      .input-group input:focus {
        outline: none;
        border-color: #fbbf24;
        box-shadow: 0 0 0 2px rgba(251, 191, 36, 0.3);
      }

      .input-group input.error {
        border-color: #ef4444;
        box-shadow: 0 0 0 2px rgba(239, 68, 68, 0.3);
      }

      .input-group i {
        position: absolute;
        left: 0.75rem;
        top: 0.85rem;
        color: #9ca3af;
        transition: all 0.3s ease;
      }

      .input-group input:focus + i {
        color: #fbbf24;
      }

      .error-message {
        background-color: rgba(239, 68, 68, 0.1);
        border: 1px solid rgba(239, 68, 68, 0.3);
        color: #fca5a5;
        padding: 0.75rem 1rem;
        border-radius: 0.5rem;
        margin-bottom: 1rem;
      }

      .success-message {
        background-color: rgba(34, 197, 94, 0.1);
        border: 1px solid rgba(34, 197, 94, 0.3);
        color: #86efac;
        padding: 0.75rem 1rem;
        border-radius: 0.5rem;
        margin-bottom: 1rem;
      }

      .error-message ul {
        margin: 0;
        padding-left: 1.25rem;
      }

      .hero-section {
        background-image: url("/api/placeholder/1920/1080");
        background-size: cover;
        background-position: center;
        position: relative;
      }

      .hero-overlay {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: linear-gradient(
          to bottom,
          rgba(15, 23, 42, 0.7),
          rgba(15, 23, 42, 0.9)
        );
      }

      .password-strength {
        height: 4px;
        border-radius: 2px;
        margin-top: 0.5rem;
        background-color: rgba(75, 85, 99, 0.5);
        overflow: hidden;
      }

      .password-strength-bar {
        height: 100%;
        border-radius: 2px;
        transition: all 0.3s ease;
        width: 0%;
      }

      .strength-weak {
        background-color: #ef4444;
        width: 25%;
      }
      .strength-fair {
        background-color: #f59e0b;
        width: 50%;
      }
      .strength-good {
        background-color: #3b82f6;
        width: 75%;
      }
      .strength-strong {
        background-color: #10b981;
        width: 100%;
      }

      /* Toast Notification Styles */
      .toast {
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 16px 20px;
        border-radius: 8px;
        color: white;
        font-weight: 500;
        z-index: 1000;
        opacity: 0;
        transform: translateX(100%);
        transition: all 0.3s ease;
        min-width: 300px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
      }

      .toast.show {
        opacity: 1;
        transform: translateX(0);
      }

      .toast-success {
        background-color: #10b981;
        border-left: 4px solid #059669;
      }

      .toast-error {
        background-color: #ef4444;
        border-left: 4px solid #dc2626;
      }

      .toast-info {
        background-color: #3b82f6;
        border-left: 4px solid #2563eb;
      }

      .toast-warning {
        background-color: #f59e0b;
        border-left: 4px solid #d97706;
      }

      /* Success Modal Styles */
      .modal-overlay {
        backdrop-filter: blur(5px);
        background-color: rgba(0, 0, 0, 0.8);
      }

      .modal-content {
        background: linear-gradient(135deg, #1f2937, #374151);
        box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.8);
      }

      .success-icon {
        background: linear-gradient(135deg, #10b981, #059669);
        animation: pulse 2s infinite;
      }

      @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.05); }
      }

      .countdown-circle {
        stroke-dasharray: 283;
        stroke-dashoffset: 283;
        animation: countdown 5s linear forwards;
      }

      @keyframes countdown {
        from { stroke-dashoffset: 283; }
        to { stroke-dashoffset: 0; }
      }

      /* Loading Animation */
      .loading-spinner {
        border: 3px solid rgba(255, 255, 255, 0.3);
        border-top: 3px solid #fbbf24;
        border-radius: 50%;
        width: 20px;
        height: 20px;
        animation: spin 1s linear infinite;
      }

      @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
      }
    </style>
  </head>

  <body class="text-white">
    <!-- Success Modal -->
    <div id="successModal" class="modal-overlay fixed inset-0 flex items-center justify-center z-50 <?php echo $show_success ? '' : 'hidden'; ?>">
      <div class="modal-content rounded-xl p-8 max-w-md mx-4 animate__animated animate__bounceIn">
        <div class="text-center">
          <div class="success-icon mx-auto flex items-center justify-center h-16 w-16 rounded-full mb-6">
            <i class="fas fa-check text-white text-2xl"></i>
          </div>
          <h3 class="text-2xl font-bold text-white mb-2">🎉 Selamat!</h3>
          <h4 class="text-lg font-semibold text-green-400 mb-4">Registrasi Berhasil!</h4>
          <p class="text-gray-300 mb-6">
            Selamat datang di KMGSTORE, <strong class="text-yellow-400"><?php echo htmlspecialchars($user_name); ?></strong>!<br>
            Akun Anda telah berhasil dibuat dan siap digunakan.
          </p>
          
          <!-- Countdown Circle -->
          <div class="relative mx-auto w-20 h-20 mb-6">
            <svg class="w-20 h-20 transform -rotate-90" viewBox="0 0 100 100">
              <circle cx="50" cy="50" r="45" stroke="rgba(75, 85, 99, 0.3)" stroke-width="8" fill="none"/>
              <circle cx="50" cy="50" r="45" stroke="#fbbf24" stroke-width="8" fill="none" 
                      class="countdown-circle" stroke-linecap="round"/>
            </svg>
            <div class="absolute inset-0 flex items-center justify-center">
              <span id="countdownNumber" class="text-2xl font-bold text-yellow-400">5</span>
            </div>
          </div>
          
          <p class="text-sm text-gray-400 mb-6">
            Anda akan dialihkan ke halaman login dalam <span id="countdownText" class="text-yellow-400 font-semibold">5</span> detik...
          </p>
          
          <div class="space-y-3">
            <button onclick="redirectToLogin()" class="w-full bg-gradient-to-r from-yellow-500 to-yellow-400 hover:from-yellow-400 hover:to-yellow-300 text-black font-bold py-3 px-6 rounded-lg transition-all duration-300 transform hover:scale-105">
              <i class="fas fa-sign-in-alt mr-2"></i>
              Lanjut ke Login Sekarang
            </button>
            <button onclick="closeModal()" class="w-full bg-gray-600 hover:bg-gray-500 text-white font-medium py-2 px-4 rounded-lg transition-all duration-300">
              <i class="fas fa-times mr-2"></i>
              Tutup & Tetap di Sini
            </button>
          </div>
        </div>
      </div>
    </div>

    <!-- Navbar -->
    <nav
      class="bg-gray-800 bg-opacity-80 backdrop-filter backdrop-blur-md sticky top-0 z-50 p-4 flex justify-between items-center shadow-lg"
    >
      <!-- Kiri: Logo -->
      <div class="flex items-center space-x-3">
        <div
          class="relative animate__animated animate__pulse animate__infinite"
        >
          <img
            src="../img/logo-tokioiboy.jpg"
            alt="Logo Tokiboy Store"
            class="h-10 w-10 rounded-full border-2 border-yellow-500"
          />
          <div
            class="absolute -top-1 -right-1 h-3 w-3 bg-green-500 rounded-full"
          ></div>
        </div>
        <div>
          <span
            class="text-xl font-bold bg-gradient-to-r from-yellow-400 via-yellow-300 to-yellow-500 text-transparent bg-clip-text"
            >KMGSTORE</span
          >
          <p class="text-xs text-gray-400">Top Up Termurah & Tercepat</p>
        </div>
      </div>

      <!-- Kanan: Menu -->
      <div class="space-x-2 flex items-center">
        <a
          href="../index.php"
          class="glow-btn bg-transparent border border-yellow-500 hover:bg-yellow-500 hover:text-black text-yellow-500 font-bold py-2 px-4 rounded-full flex items-center transition-colors"
        >
          <i class="fas fa-home mr-1"></i> Beranda
        </a>
        <a
          href="masuk.php"
          class="glow-btn bg-blue-500 hover:bg-blue-400 text-white font-bold py-2 px-4 rounded-full flex items-center"
        >
          <i class="fas fa-sign-in-alt mr-1"></i> Masuk
        </a>
      </div>
    </nav>

    <!-- Hero Section -->
    <section class="hero-section py-16 relative mb-8">
      <div class="hero-overlay"></div>
      <div class="container mx-auto px-4 relative z-10">
        <div class="max-w-xl mx-auto text-center">
          <h1
            class="text-4xl font-bold mb-4 animate__animated animate__fadeInDown"
          >
            <span
              class="bg-gradient-to-r from-yellow-400 via-yellow-300 to-yellow-500 text-transparent bg-clip-text"
              >Bergabung dengan KMGSTORE</span
            >
          </h1>
          <p
            class="text-lg mb-6 text-gray-300 animate__animated animate__fadeInUp"
          >
            Daftar sekarang dan nikmati top up games favorit dengan harga
            termurah dan proses tercepat!
          </p>
        </div>
      </div>
    </section>

    <!-- Registration Form Section -->
    <section class="mb-12 px-4">
      <div class="container mx-auto max-w-md">
        <div
          class="register-card gradient-border bg-gray-800 bg-opacity-70 backdrop-filter backdrop-blur-sm rounded-xl overflow-hidden p-8 shadow-xl"
        >
          <div class="text-center mb-6">
            <div
              class="inline-block rounded-full p-3 bg-yellow-500 bg-opacity-20 mb-4"
            >
              <i class="fas fa-user-plus text-yellow-500 text-3xl"></i>
            </div>
            <h2 class="text-2xl font-bold">Daftar Akun Baru</h2>
            <p class="text-gray-400 text-sm mt-1">
              Buat akun Anda untuk mulai top up games
            </p>
          </div>

          <?php if (!empty($errors)): ?>
          <div class="error-message animate__animated animate__shakeX">
            <i class="fas fa-exclamation-triangle mr-2"></i>
            <strong>Terjadi kesalahan:</strong>
            <ul>
              <?php foreach ($errors as $error): ?>
              <li><?php echo htmlspecialchars($error); ?></li>
              <?php endforeach; ?>
            </ul>
          </div>
          <?php endif; ?>

          <form action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" method="POST" id="registerForm">
            <div class="input-group">
              <input
                type="text"
                id="nama"
                name="nama"
                placeholder="Nama Lengkap"
                value="<?php echo isset($form_data['nama']) ? htmlspecialchars($form_data['nama']) : ''; ?>"
                required
                minlength="2"
                maxlength="100"
              />
              <i class="fas fa-user"></i>
            </div>

            <div class="input-group">
              <input
                type="email"
                id="email"
                name="email"
                placeholder="Alamat Email"
                value="<?php echo isset($form_data['email']) ? htmlspecialchars($form_data['email']) : ''; ?>"
                required
                maxlength="100"
              />
              <i class="fas fa-envelope"></i>
            </div>

            <div class="input-group">
              <input
                type="tel"
                id="nohp"
                name="nohp"
                placeholder="Nomor HP (Contoh: 08123456789)"
                value="<?php echo isset($form_data['nohp']) ? htmlspecialchars($form_data['nohp']) : ''; ?>"
                required
                pattern="^0[0-9]{9,12}$"
                title="Nomor HP harus dimulai dengan 0 dan memiliki 10-13 digit"
              />
              <i class="fas fa-phone"></i>
            </div>

            <div class="input-group">
              <input
                type="password"
                id="password"
                name="password"
                placeholder="Kata Sandi"
                required
                minlength="8"
              />
              <i class="fas fa-lock"></i>
              <div class="password-strength">
                <div class="password-strength-bar" id="strengthBar"></div>
              </div>
              <p class="text-xs text-gray-400 mt-1" id="strengthText">
                Minimal 8 karakter, gunakan kombinasi huruf, angka, dan simbol
              </p>
            </div>

            <div class="input-group">
              <input
                type="password"
                id="confirmPassword"
                name="confirmPassword"
                placeholder="Konfirmasi Kata Sandi"
                required
                minlength="8"
              />
              <i class="fas fa-lock"></i>
              <p class="text-xs text-red-400 mt-1 hidden" id="confirmPasswordError">
                Password tidak cocok
              </p>
            </div>

            <div class="flex items-start mb-6">
              <input
                type="checkbox"
                id="terms"
                class="h-4 w-4 rounded border-gray-600 focus:ring-yellow-500 mt-1"
                required
              />
              <label for="terms" class="ml-2 text-sm text-gray-400">
                Saya setuju dengan
                <a href="#tos" class="text-yellow-500 hover:underline"
                  >Syarat & Ketentuan</a
                >
                dan
                <a href="#privacy" class="text-yellow-500 hover:underline"
                  >Kebijakan Privasi</a
                >
              </label>
            </div>

            <button
              type="submit"
              class="w-full glow-btn bg-yellow-500 hover:bg-yellow-400 text-black font-bold py-3 px-4 rounded-lg transition flex items-center justify-center"
              id="submitBtn"
            >
              <i class="fas fa-user-plus mr-2"></i> Daftar Sekarang
            </button>
          </form>

          <div class="mt-6 text-center">
            <p class="text-gray-400"><p class="text-gray-400">
              Sudah punya akun?
              <a href="masuk.php" class="text-yellow-500 hover:underline font-medium">
                Masuk di sini
              </a>
            </p>
          </div>
        </div>
      </div>
    </section>

    <!-- Footer -->
    <footer class="bg-gray-800 bg-opacity-80 backdrop-filter backdrop-blur-md py-8 mt-12">
      <div class="container mx-auto px-4 text-center">
        <div class="flex justify-center items-center mb-4">
          <img
            src="../img/logo-tokioiboy.jpg"
            alt="Logo KMGSTORE"
            class="h-8 w-8 rounded-full mr-2"
          />
          <span class="text-xl font-bold bg-gradient-to-r from-yellow-400 to-yellow-500 text-transparent bg-clip-text">
            KMGSTORE
          </span>
        </div>
        <p class="text-gray-400 text-sm mb-4">
          Platform top up games terpercaya dengan harga termurah dan proses tercepat
        </p>
        <div class="flex justify-center space-x-4 mb-4">
          <a href="#" class="text-gray-400 hover:text-yellow-500 transition">
            <i class="fab fa-facebook-f"></i>
          </a>
          <a href="#" class="text-gray-400 hover:text-yellow-500 transition">
            <i class="fab fa-instagram"></i>
          </a>
          <a href="#" class="text-gray-400 hover:text-yellow-500 transition">
            <i class="fab fa-twitter"></i>
          </a>
          <a href="#" class="text-gray-400 hover:text-yellow-500 transition">
            <i class="fab fa-whatsapp"></i>
          </a>
        </div>
        <p class="text-gray-500 text-xs">
          © 2024 KMGSTORE. Semua hak dilindungi.
        </p>
      </div>
    </footer>

    <script>
      // Password strength checker
      const passwordInput = document.getElementById('password');
      const strengthBar = document.getElementById('strengthBar');
      const strengthText = document.getElementById('strengthText');
      const confirmPasswordInput = document.getElementById('confirmPassword');
      const confirmPasswordError = document.getElementById('confirmPasswordError');
      const submitBtn = document.getElementById('submitBtn');
      const registerForm = document.getElementById('registerForm');

      // Password strength checking
      passwordInput.addEventListener('input', function() {
        const password = this.value;
        const strength = checkPasswordStrength(password);
        
        strengthBar.className = 'password-strength-bar ' + strength.class;
        strengthText.textContent = strength.message;
        strengthText.className = 'text-xs mt-1 ' + strength.color;
      });

      // Confirm password validation
      confirmPasswordInput.addEventListener('input', function() {
        const password = passwordInput.value;
        const confirmPassword = this.value;
        
        if (confirmPassword && password !== confirmPassword) {
          confirmPasswordError.classList.remove('hidden');
          this.classList.add('error');
        } else {
          confirmPasswordError.classList.add('hidden');
          this.classList.remove('error');
        }
      });

      // Form submission with loading state
      registerForm.addEventListener('submit', function(e) {
        // Show loading state
        submitBtn.innerHTML = '<div class="loading-spinner mr-2"></div> Mendaftar...';
        submitBtn.disabled = true;
        
        // Validate form before submission
        const password = passwordInput.value;
        const confirmPassword = confirmPasswordInput.value;
        
        if (password !== confirmPassword) {
          e.preventDefault();
          showToast('Password dan konfirmasi password tidak cocok!', 'error');
          resetSubmitButton();
          return;
        }
        
        // Additional client-side validation can be added here
      });

      function checkPasswordStrength(password) {
        let strength = 0;
        let message = '';
        let className = '';
        let color = '';
        
        if (password.length >= 8) strength++;
        if (password.match(/[a-z]/)) strength++;
        if (password.match(/[A-Z]/)) strength++;
        if (password.match(/[0-9]/)) strength++;
        if (password.match(/[^A-Za-z0-9]/)) strength++;
        
        switch(strength) {
          case 0:
          case 1:
            message = 'Password terlalu lemah';
            className = 'strength-weak';
            color = 'text-red-400';
            break;
          case 2:
            message = 'Password lemah';
            className = 'strength-weak';
            color = 'text-red-400';
            break;
          case 3:
            message = 'Password sedang';
            className = 'strength-fair';
            color = 'text-yellow-400';
            break;
          case 4:
            message = 'Password kuat';
            className = 'strength-good';
            color = 'text-blue-400';
            break;
          case 5:
            message = 'Password sangat kuat';
            className = 'strength-strong';
            color = 'text-green-400';
            break;
        }
        
        return { class: className, message: message, color: color };
      }

      function resetSubmitButton() {
        submitBtn.innerHTML = '<i class="fas fa-user-plus mr-2"></i> Daftar Sekarang';
        submitBtn.disabled = false;
      }

      function showToast(message, type = 'info') {
        const toast = document.createElement('div');
        toast.className = `toast toast-${type}`;
        toast.innerHTML = `
          <div class="flex items-center">
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'} mr-2"></i>
            <span>${message}</span>
          </div>
        `;
        
        document.body.appendChild(toast);
        
        setTimeout(() => {
          toast.classList.add('show');
        }, 100);
        
        setTimeout(() => {
          toast.classList.remove('show');
          setTimeout(() => {
            document.body.removeChild(toast);
          }, 300);
        }, 3000);
      }

      // Success modal functions
      function closeModal() {
        document.getElementById('successModal').classList.add('hidden');
      }

      function redirectToLogin() {
        window.location.href = 'masuk.php';
      }

      // Auto redirect countdown
      <?php if ($show_success): ?>
      let countdown = 5;
      const countdownInterval = setInterval(function() {
        countdown--;
        document.getElementById('countdownNumber').textContent = countdown;
        document.getElementById('countdownText').textContent = countdown;
        
        if (countdown <= 0) {
          clearInterval(countdownInterval);
          redirectToLogin();
        }
      }, 1000);
      <?php endif; ?>

      // Phone number formatting
      document.getElementById('nohp').addEventListener('input', function(e) {
        let value = e.target.value.replace(/[^0-9]/g, '');
        if (value.length > 0 && !value.startsWith('0')) {
          value = '0' + value;
        }
        if (value.length > 13) {
          value = value.substring(0, 13);
        }
        e.target.value = value;
      });

      // Real-time validation
      document.addEventListener('DOMContentLoaded', function() {
        const inputs = document.querySelectorAll('input[required]');
        
        inputs.forEach(input => {
          input.addEventListener('blur', function() {
            if (this.checkValidity()) {
              this.classList.remove('error');
            } else {
              this.classList.add('error');
            }
          });
        });
      });

      // Prevent form resubmission on refresh
      if (window.history.replaceState) {
        window.history.replaceState(null, null, window.location.href);
      }
    </script>
  </body>
</html></p>