<?php
session_start();
require_once "config.php";
require_once "functions.php";

// Check user authentication
cek_login_user();

if (!isset($_SESSION['user_id'])) {
    header("Location: masuk.html");
    exit();
}

$user_id = $_SESSION['user_id'];

// Pagination setup
$limit = 10;
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * $limit;

// Filters
$status_filter = $_GET['status'] ?? '';
$game_filter = $_GET['game'] ?? '';
$search_filter = $_GET['search'] ?? '';

// Build WHERE clause for database query
$where_conditions = ["user_id = ?"];
$params = [$user_id];
$param_types = "s";

if (!empty($status_filter)) {
    $where_conditions[] = "status = ?";
    $params[] = $status_filter;
    $param_types .= "s";
}

if (!empty($game_filter)) {
    $where_conditions[] = "nama_game LIKE ?";
    $params[] = "%$game_filter%";
    $param_types .= "s";
}

if (!empty($search_filter)) {
    $where_conditions[] = "(kode_transaksi LIKE ? OR userid_game LIKE ?)";
    $params[] = "%$search_filter%";
    $params[] = "%$search_filter%";
    $param_types .= "ss";
}

$where_clause = "WHERE " . implode(" AND ", $where_conditions);

// Get total count for pagination
$total_records = 0;
$total_pages = 0;

try {
    $count_sql = "SELECT COUNT(*) as total FROM transaksi $where_clause";
    $count_stmt = $conn->prepare($count_sql);
    if (!empty($params)) {
        $count_stmt->bind_param($param_types, ...$params);
    }
    $count_stmt->execute();
    $total_result = $count_stmt->get_result();
    $total_row = $total_result->fetch_assoc();
    $total_records = $total_row['total'];
    $total_pages = ceil($total_records / $limit);
} catch (Exception $e) {
    error_log("Error fetching total records: " . $e->getMessage());
}

// Get transactions
$transaksi_list = [];

try {
    $sql = "SELECT * FROM transaksi $where_clause ORDER BY tanggal DESC LIMIT ? OFFSET ?";
    $stmt = $conn->prepare($sql);
    
    // Add limit and offset to params
    $params[] = $limit;
    $params[] = $offset;
    $param_types .= "ii";
    
    $stmt->bind_param($param_types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    $transaksi_list = $result->fetch_all(MYSQLI_ASSOC);
} catch (Exception $e) {
    error_log("Error fetching transactions: " . $e->getMessage());
}

// Helper functions
function getStatusBadge($status) {
    $badges = [
        'berhasil' => '<span class="bg-green-100 text-green-800 text-xs font-medium px-2.5 py-0.5 rounded">✓ Berhasil</span>',
        'pending' => '<span class="bg-yellow-100 text-yellow-800 text-xs font-medium px-2.5 py-0.5 rounded">⏳ Pending</span>',
        'gagal' => '<span class="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">✗ Gagal</span>'
    ];
    
    return $badges[strtolower($status)] ?? '<span class="bg-gray-100 text-gray-800 text-xs font-medium px-2.5 py-0.5 rounded">' . ucfirst($status) . '</span>';
}

function formatRupiah($angka) {
    if (is_string($angka)) {
        $angka = preg_replace('/[^0-9.]/', '', $angka);
    }
    
    $angka = intval(floatval($angka));
    return "Rp " . number_format($angka, 0, ',', '.');
}

function extractItemInfo($nominal) {
    $item_text = preg_replace('/Rp\s+[\d.,]+/', '', $nominal);
    return trim($item_text);
}

function buildPaginationUrl($page, $status_filter, $game_filter, $search_filter) {
    $params = [
        'page' => $page,
        'status' => $status_filter,
        'game' => $game_filter,
        'search' => $search_filter
    ];
    
    return '?' . http_build_query(array_filter($params));
}

$games_list = [
    'Mobile Legends',
    'PUBG Mobile',
    'Free Fire',
    'Genshin Impact',
    'Honor of Kings',
    'Valorant'
];
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cek Pesanan - KMGSTORE</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'primary': '#1e2635',
                        'secondary': '#2a2a2a',
                        'accent': '#daa520'
                    }
                }
            }
        }
    </script>
    <style>
        body {
            background: #1e1e1e;
            color: #fff;
            font-family: 'Inter', sans-serif;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }
        .glass-effect {
            background: rgba(42, 42, 42, 0.8);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        .hover-scale {
            transition: transform 0.2s ease;
        }
        .hover-scale:hover {
            transform: scale(1.02);
        }
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
        }
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        .main-content {
            flex: 1;
            min-height: calc(100vh - 200px);
        }

        .footer-bottom {
      background-color: #1e2635;
      color: white;
      padding: 20px;
      text-align: center;
      font-size: 13px;
      margin-top: 40px;
    }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="bg-gray-800 bg-opacity-80 backdrop-filter backdrop-blur-md sticky top-0 z-50 p-4 flex justify-between items-center shadow-lg">
        <div class="flex items-center space-x-3">
            <div class="relative">
                <img src="../img/logo-tokioiboy.jpg" alt="Logo Tokiboy Store" class="h-10 w-10 rounded-full border-2 border-yellow-500" />
                <div class="absolute -top-1 -right-1 h-3 w-3 bg-green-500 rounded-full"></div>
            </div>
            <div>
                <span class="text-xl font-bold bg-gradient-to-r from-yellow-400 via-yellow-300 to-yellow-500 text-transparent bg-clip-text">KMGSTORE</span>
                <p class="text-xs text-gray-400">Top Up Termurah & Tercepat</p>
            </div>
        </div>

        <div class="space-x-2 flex items-center">
            <?php if (isset($_SESSION['nama'])): ?>
                <div class="bg-gray-700 px-3 py-1 rounded-full flex items-center">
                    <i class="fas fa-user-circle mr-2"></i>
                    <span class="text-white">Halo, <?= htmlspecialchars($_SESSION['nama']) ?>!</span>
                </div>
                <a href="../index.php" class="bg-red-500 hover:bg-red-600 text-white font-bold py-2 px-4 rounded-full flex items-center">
                    <i class="fas fa-home mr-1"></i> Home
                </a>
            <?php else: ?>
                <a href="masuk.php" class="bg-yellow-500 text-black px-3 py-1 rounded">Masuk</a>
                <a href="daftar.php" class="bg-gray-700 px-3 py-1 rounded border border-yellow-500">Daftar</a>
            <?php endif; ?>
        </div>
    </nav>

    <!-- Main Content -->
    <div class="main-content">
        <div class="max-w-7xl mx-auto px-4 py-8">
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold mb-2">Riwayat Pesanan</h1>
                <p class="text-gray-400">Lihat semua transaksi dan pesanan Anda</p>
            </div>

            <!-- Filters -->
            <div class="glass-effect rounded-xl p-6 mb-6">
                <form method="GET" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <div>
                        <label class="block text-sm font-medium mb-2">Cari Transaksi</label>
                        <input type="text" name="search" value="<?= htmlspecialchars($search_filter) ?>" 
                               placeholder="ID Transaksi atau User ID" 
                               class="w-full px-3 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white placeholder-gray-400 focus:ring-2 focus:ring-accent focus:border-transparent">
                    </div>

                    <div>
                        <label class="block text-sm font-medium mb-2">Status</label>
                        <select name="status" class="w-full px-3 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:ring-2 focus:ring-accent focus:border-transparent">
                            <option value="">Semua Status</option>
                            <option value="berhasil" <?= $status_filter === 'berhasil' ? 'selected' : '' ?>>Berhasil</option>
                            <option value="pending" <?= $status_filter === 'pending' ? 'selected' : '' ?>>Pending</option>
                            <option value="gagal" <?= $status_filter === 'gagal' ? 'selected' : '' ?>>Gagal</option>
                        </select>
                    </div>

                    <div>
                        <label class="block text-sm font-medium mb-2">Game</label>
                        <select name="game" class="w-full px-3 py-2 bg-gray-700 border border-gray-600 rounded-lg text-white focus:ring-2 focus:ring-accent focus:border-transparent">
                            <option value="">Semua Game</option>
                            <?php foreach ($games_list as $game): ?>
                                <option value="<?= $game ?>" <?= $game_filter === $game ? 'selected' : '' ?>><?= $game ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="flex items-end">
                        <button type="submit" class="w-full bg-accent hover:bg-yellow-600 text-black px-4 py-2 rounded-lg font-medium transition">
                            Filter
                        </button>
                    </div>
                </form>
            </div>

            <!-- Results Summary -->
            <div class="mb-6">
                <p class="text-gray-400">
                    Menampilkan <?= count($transaksi_list) ?> dari <?= $total_records ?> transaksi
                    <?php if (!empty($search_filter) || !empty($status_filter) || !empty($game_filter)): ?>
                        <a href="cekpesanan.php" class="text-accent hover:underline ml-2">Reset Filter</a>
                    <?php endif; ?>
                </p>
            </div>

            <!-- Transaction List -->
            <?php if (empty($transaksi_list)): ?>
                <div class="glass-effect rounded-xl p-12 text-center mb-12">
                    <div class="text-6xl mb-4">📦</div>
                    <h3 class="text-xl font-semibold mb-2">Tidak Ada Transaksi</h3>
                    <p class="text-gray-400 mb-6">
                        <?php if (!empty($search_filter) || !empty($status_filter) || !empty($game_filter)): ?>
                            Tidak ada transaksi yang sesuai dengan filter Anda.
                        <?php else: ?>
                            Anda belum memiliki transaksi apapun. Yuk mulai berbelanja!
                        <?php endif; ?>
                    </p>
                    <a href="index.php" class="bg-accent hover:bg-yellow-600 text-black px-6 py-3 rounded-lg font-medium transition inline-block">
                        Mulai Belanja
                    </a>
                </div>
            <?php else: ?>
                <div class="space-y-4 mb-12">
                    <?php foreach ($transaksi_list as $transaksi): ?>
                        <div class="glass-effect rounded-xl p-6 hover-scale">
                            <div class="flex flex-col md:flex-row md:items-center justify-between mb-4">
                                <div class="flex items-center space-x-3 mb-2 md:mb-0">
                                    <div class="w-10 h-10 bg-accent rounded-lg flex items-center justify-center text-black font-bold text-sm">
                                        🎮
                                    </div>
                                    <div>
                                        <h3 class="font-semibold text-lg"><?= htmlspecialchars($transaksi['nama_game']) ?></h3>
                                        <p class="text-sm text-gray-400">ID: <?= htmlspecialchars($transaksi['kode_transaksi']) ?></p>
                                    </div>
                                </div>
                                <div class="flex items-center space-x-3">
                                    <?= getStatusBadge($transaksi['status']) ?>
                                    <span class="text-sm text-gray-400"><?= date('d/m/Y H:i', strtotime($transaksi['tanggal'])) ?></span>
                                </div>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                                <div>
                                    <p class="text-sm text-gray-400">User ID Game</p>
                                    <p class="font-medium"><?= htmlspecialchars($transaksi['userid_game']) ?><?= !empty($transaksi['zoneid_game']) ? " (" . htmlspecialchars($transaksi['zoneid_game']) . ")" : "" ?></p>
                                </div>
                                <div>
                                    <p class="text-sm text-gray-400">Item</p>
                                    <p class="font-medium">
                                        <?php 
                                        if (!empty($transaksi['item_jumlah']) && !empty($transaksi['item_jenis'])) {
                                            echo htmlspecialchars($transaksi['item_jumlah'] . ' ' . $transaksi['item_jenis']);
                                        } else {
                                            echo htmlspecialchars(extractItemInfo($transaksi['nominal']));
                                        }
                                        ?>
                                    </p>
                                </div>
                                <div>
                                    <p class="text-sm text-gray-400">Metode Pembayaran</p>
                                    <p class="font-medium"><?= htmlspecialchars($transaksi['metode_pembayaran']) ?></p>
                                </div>
                            </div>

                            <div class="flex justify-between items-center pt-4 border-t border-gray-600">
                                <div>
                                    <p class="text-sm text-gray-400">Total Pembayaran</p>
                                    <p class="text-xl font-bold text-accent">
                                        <?php 
                                        if (!empty($transaksi['total_bayar']) && $transaksi['total_bayar'] != '0') {
                                            echo formatRupiah($transaksi['total_bayar']);
                                        } else {
                                            $nominal_number = preg_replace('/[^0-9]/', '', $transaksi['nominal']);
                                            echo formatRupiah($nominal_number);
                                        }
                                        ?>
                                    </p>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="mb-12 flex justify-center">
                        <nav class="flex space-x-2">
                            <?php if ($page > 1): ?>
                                <a href="<?= buildPaginationUrl($page - 1, $status_filter, $game_filter, $search_filter) ?>" 
                                   class="px-3 py-2 bg-gray-700 hover:bg-gray-600 rounded-lg text-sm transition">
                                    Sebelumnya
                                </a>
                            <?php endif; ?>

                            <?php
                            $start_page = max(1, $page - 2);
                            $end_page = min($total_pages, $page + 2);

                            for ($i = $start_page; $i <= $end_page; $i++):
                            ?>
                                <a href="<?= buildPaginationUrl($i, $status_filter, $game_filter, $search_filter) ?>" 
                                   class="px-3 py-2 <?= $i === $page ? 'bg-accent text-black' : 'bg-gray-700 hover:bg-gray-600' ?> rounded-lg text-sm transition">
                                    <?= $i ?>
                                </a>
                            <?php endfor; ?>

                            <?php if ($page < $total_pages): ?>
                                <a href="<?= buildPaginationUrl($page + 1, $status_filter, $game_filter, $search_filter) ?>" 
                                   class="px-3 py-2 bg-gray-700 hover:bg-gray-600 rounded-lg text-sm transition">
                                    Selanjutnya
                                </a>
                            <?php endif; ?>
                        </nav>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>

    <!-- Detail Modal -->
    <div id="detailModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
        <div class="glass-effect rounded-xl max-w-2xl w-full max-h-screen overflow-y-auto">
            <div class="p-6">
                <div class="flex justify-between items-center mb-6">
                    <h3 class="text-xl font-bold">Detail Transaksi</h3>
                    <button onclick="closeModal()" class="text-gray-400 hover:text-white transition">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div id="detailContent">
                    <div class="loading"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Footer -->
      <div class="footer-bottom">© 2025 KMGSTORE. All Rights Reserved.</div>


    <script>
        function showDetail(kodeTransaksi) {
            const modal = document.getElementById('detailModal');
            const content = document.getElementById('detailContent');

            modal.classList.remove('hidden');
            content.innerHTML = '<div class="loading"></div>';

            fetch('get_transaction_detail.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'kode_transaksi=' + encodeURIComponent(kodeTransaksi)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    content.innerHTML = generateDetailHTML(data.transaksi);
                } else {
                    content.innerHTML = '<p class="text-red-400">Error: ' + data.message + '</p>';
                }
            })
            .catch(error => {
                content.innerHTML = '<p class="text-red-400">Terjadi kesalahan saat memuat data.</p>';
                console.error('Error:', error);
            });
        }

        function generateDetailHTML(transaksi) {
            const statusBadge = getStatusBadgeJS(transaksi.status);
            const tanggal = new Date(transaksi.tanggal).toLocaleString('id-ID');

            let itemInfo = '';
            if (transaksi.item_jumlah && transaksi.item_jenis) {
                itemInfo = transaksi.item_jumlah + ' ' + transaksi.item_jenis;
            } else if (transaksi.nominal) {
                itemInfo = transaksi.nominal.replace(/Rp\s+[\d.,]+/g, '').trim();
            }

            let totalAmount = '0';
            if (transaksi.total_bayar && transaksi.total_bayar != '0') {
                totalAmount = parseInt(transaksi.total_bayar.toString().replace(/[^0-9]/g, '')).toLocaleString('id-ID');
            } else if (transaksi.nominal) {
                const nominalNumber = transaksi.nominal.replace(/[^0-9]/g, '');
                if (nominalNumber) {
                    totalAmount = parseInt(nominalNumber).toLocaleString('id-ID');
                }
            }

            return `
                <div class="space-y-4">
                    <div class="bg-gray-800 bg-opacity-50 p-4 rounded-lg">
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <p class="text-sm text-gray-400">ID Transaksi</p>
                                <p class="font-mono">${transaksi.kode_transaksi}</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-400">Tanggal</p>
                                <p>${tanggal}</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-400">Game</p>
                                <p class="font-medium">${transaksi.nama_game}</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-400">Status</p>
                                ${statusBadge}
                            </div>
                            <div>
                                <p class="text-sm text-gray-400">User ID Game</p>
                                <p>${transaksi.userid_game}${transaksi.zoneid_game ? ' (' + transaksi.zoneid_game + ')' : ''}</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-400">Item</p>
                                <p>${itemInfo}</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-400">Metode Pembayaran</p>
                                <p>${transaksi.metode_pembayaran}</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-400">Total</p>
                                <p class="text-lg font-bold text-accent">Rp ${totalAmount}</p>
                            </div>
                        </div>
                    </div>
                    ${transaksi.whatsapp ? `
                    <div class="bg-blue-900 bg-opacity-30 border border-blue-500 rounded-lg p-4">
                        <p class="text-blue-300">📱 WhatsApp: ${transaksi.whatsapp}</p>
                    </div>` : ''}
                </div>
            `;
        }

        function getStatusBadgeJS(status) {
            const badges = {
                'berhasil': '<span class="bg-green-100 text-green-800 text-xs font-medium px-2.5 py-0.5 rounded">✓ Berhasil</span>',
                'pending': '<span class="bg-yellow-100 text-yellow-800 text-xs font-medium px-2.5 py-0.5 rounded">⏳ Pending</span>',
                'gagal': '<span class="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">✗ Gagal</span>'
            };
            
            return badges[status.toLowerCase()] || `<span class="bg-gray-100 text-gray-800 text-xs font-medium px-2.5 py-0.5 rounded">${status}</span>`;
        }

        function closeModal() {
            document.getElementById('detailModal').classList.add('hidden');
        }

        function downloadStruk(kodeTransaksi) {
            window.open('download_struk.php?kode=' + encodeURIComponent(kodeTransaksi), '_blank');
        }

        // Close modal when clicking outside
        document.getElementById('detailModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
    </script>
</body>
</html>